%   #######################################################################
%   Implementing Finite Difference method for contingent claim pricing  
%   Created by: Nik Tuzov, www.ntuzov.com
%   #######################################################################
 
%   #######################################################################
%   Example1: a standard European call. Initial data:
    
     S = 110;            % stock price, current
     K = 100;            % strike price
     r = 0.05;           % risk - free rate
     sigma = 0.3;        % volatility
     T = 1;              % time to maturity     
     theta = 0.5;        % theta = 1: fully implicit method
                         % theta = 0: fully explicit method
                         % theta = 0.5: Crank and Nicholson scheme     
 %   ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ 
     % True call price, Black-Scholes-based: 
       [BS_call, BS_put]  = blsprice(S, K, r, T, sigma);
    
      % L, state space localization parameter:      
      L = max( abs(log(S) + (r - sigma^2 /2) + 4 * sigma * sqrt(T)), abs(log(S) + (r - sigma^2 /2) - 4 * sigma * sqrt(T)) );     
      
      max_space =  500; % maximal number of space nodes  
      max_time = 60;    % maximal number of time nodes 
      
      % steps for the number of time and space nodes:
      step_time = 10;     
      step_space = 20;      
      
      % grid for time and space
      [space, time] = meshgrid([step_space : step_space : max_space], [step_time : step_time : max_time]);   
      
      % calculating the pricing error:
      value = space;            
      for i = 1 : size(value,1)
          for j = 1 : size(value, 2)
              value(i,j) = abs( BS_call - claim(@call, 'euro', S, r, sigma, T, L, space(i, j), time(i, j), theta));          
          end;
      end;              
      
      % 3D plot: shows the convergence as the number of time and space
      % nodes go up      
      subplot(3,2, 1)
      mesh(space, time, value),
      xlabel('N, number of space nodes'), ylabel('M, number of time nodes'), 
      zlabel('Euro-call option, pricing error, $'), title('Convergence VS N and M'), grid on;
      
     % 2D plot: shows the goodness of estimation w.r.t. theta 
      N = 500;
      M = 1000;     
      i = 1;
      theta_step = 0.1;
      graph = zeros(round(1 / theta_step) + 1, 2);
      
      for theta = 0 : theta_step : 1
          graph(i, 1) = theta;
          graph(i, 2) = abs( BS_call - claim(@call, 'euro', S, r, sigma, T, L, N, M, theta));  
          i = i + 1;
      end;    
      subplot(3,2, 2)
      plot(graph(:, 1), graph(:, 2)), xlabel(' theta '), ylabel('Euro-call option, pricing error, $'), 
           title('Convergence VS theta'), grid;       
      
%  #######################################################################  
%   Example2: an Exotic European option. Initial data are the same as for
%   Example1.
%   ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++      
      theta = 0.5;   
      % calculating the price estimate:
      price = space;            
      for i = 1 : size(price,1)
          for j = 1 : size(price, 2)
              price(i,j) = claim(@root, 'euro', S, r, sigma, T, L, space(i, j), time(i, j), theta);                
          end;
      end;      
      % 3D plot: shows the convergence as the number of time and space
      % nodes go up
      subplot(3,2, 3)
      mesh(space, time, price),
      xlabel('N, number of space nodes'), ylabel('M, number of time nodes'), 
      zlabel('European "root" option, price, $'), title('Convergence VS N and M'), grid on;
      
%  ####################################################################### %
%   Example3: an standard American put. Initial data are the same 
%   as for Example1     
%   ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++     
     theta = 1;    
     % calculating the price estimate:
      price = space;            
      for i = 1 : size(price,1)
          for j = 1 : size(price, 2)
              price(i,j) = claim(@put, 'amer', S, r, sigma, T, L, space(i, j), time(i, j), theta);                           
          end;
      end;      
      % 3D plot: shows the convergence as the number of time and space
      % nodes go up      
      subplot(3,2, 4)
      mesh(space, time, price),
      xlabel('N, number of space nodes'), ylabel('M, number of time nodes'), 
      zlabel('American put option, price, $'), title('Convergence VS N and M'), grid on;

% %  ####################################################################### 
% %   Example4: an exotic  American "sine" option. Initial data are the same 
%     as for Example1     
% %   ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++     
     theta = 1;    
     % calculating the price estimate:
      price = space;            
      for i = 1 : size(price,1)
          for j = 1 : size(price, 2)
              price(i,j) = claim(@sine_opt, 'amer', S, r, sigma, T, L, space(i, j), time(i, j), theta);                            
          end;
      end;      
      % 3D plot: shows the convergence as the number of time and space
      % nodes go up      
      subplot(3,2, 5)
      mesh(space, time, price),
      xlabel('N, number of space nodes'), ylabel('M, number of time nodes'), 
      zlabel('American "sine" option, price, $'), title('Convergence VS N and M'), grid on;
      
% %  ####################################################################### 
      
    